import Image from "next/image";
import heroImage from "@/public/assets/hero.png";
import { Button } from "@mantine/core";
import { AnimatePresence, motion } from "framer-motion";
import { useState, useEffect } from "react";
import { useTranslation } from "react-i18next";

const textVariants = {
  hidden: { opacity: 0, y: 20 },
  visible: { opacity: 1, y: 0 },
  exit: { opacity: 0, y: -20 },
};

const Hero = () => {
  const { t, i18n } = useTranslation();
  const currentLanguage = i18n.language;
  const textDirectionClass =
    currentLanguage === "ar"
      ? "lg:items-end lg:text-right"
      : "lg:items-start lg:text-left";
      
  const textDir = currentLanguage === 'ar' ? 'rtl' : 'ltr';

  const [currentIndex, setCurrentIndex] = useState(0);
  const texts: string[] = t("hero.animationTitle", {
    returnObjects: true,
  }) as string[];

  useEffect(() => {
    if (texts.length > 0) {
      const interval = setInterval(() => {
        setCurrentIndex((prevIndex) => (prevIndex + 1) % texts.length);
      }, 3000);

      return () => clearInterval(interval);
    }
  }, [texts.length]);

  return (
    <section className="relative overflow-hidden hero h-screen grid grid-cols-1 lg:grid-cols-2 lg:grid-rows-1 grid-rows-2">
      <div className="relative row-start-1 lg:row-start-auto b-10">
        <Image
          src={heroImage}
          alt="hero"
          priority
          className="object-cover min-w-[140vw] md:min-w-full md:h-full lg:mask-none mask-image-linear translate-y-[-120px] md:translate-y-0"
        />
      </div>

      <div
        className={`flex flex-col z-20 lg:w-auto row-start-2 lg:row-start-auto lg:justify-center text-right p-4 lg:mr-16 mt-0 xs:mt-11 lg:mt-0 lg:pt-20 ${textDirectionClass}`}
      >
        <h2 className="text-[20px] md:text-[36px] lg:text-[48px] text-[#0C3882] pb-0 md:mb-7 font-bold font-laxr leading-[50px]">
          {t("hero.title")}
        </h2>
        <AnimatePresence mode="wait" initial={false}>
          {texts.length > 0 && (
            <motion.div
              key={currentIndex}
              variants={textVariants}
              initial="hidden"
              animate="visible"
              exit="exit"
              transition={{ duration: 1 }}
              className=" bg-gradient-to-r from-[#0C3580] to-[#1BBBCF] w-full bg-clip-text text-transparent md:py-3 py-1 "
            >
              <span className="px-2 text-[20px] md:text-[36px] lg:text-[42px] font-bold font-laxr leading-[30px] lg:leading-[50px]">{texts[currentIndex]}</span>
            </motion.div>
          )}
        </AnimatePresence>
        <div className="mt-4 md:mt-6 lg:mt-11">
          <div className="text-[#0C3882] text-[11px] md:text-[14px] md:text-base font-medium font-cairo lg:text-lg lg:max-w-[550px]" dir={textDir}>
            <p>{t("hero.subTitle")}</p>
          </div>
          <Button
            component="a"
            href="#contact-us"
            className="mt-4 sm:mt-4 bg-gradient-to-r from-[#0C3580] to-[#1BBBCF] border-none rounded-xl px-9 h-12"
            color="blue"
          >
            <span className="px-1 font-laxr text-base">{t("hero.button")}</span>
          </Button>
        </div>
      </div>
    </section>
  );
};

export default Hero;
